--The scripts should always contain at least 10 functions :ScriptID, DisplaySourceName(), SourceSuperClassID(), SourceClassID(), DisplayDestinationName() DestinationSuperClassID(), DestinationClassID(), AboutText() and DefaultConversion, conversion taking a param.
--Script ID that will append to destination
fn ScriptID = 
(
	""
)

--Return the name to be display for the source in the Scene Converter UI - must be in the order than the source class ids
fn DisplaySourceName =
(
	#("Photometric Free Light",
		"Photometric Target Light",
		"Photometric Free Disc",
		"Photometric Target Disc",
		"Photometric Free Rectangle",
		"Photometric Target Rectangle",
		"Photometric Free Cyclinder",
		"Photometric Target Cylinder",
		"Photometric Free Sphere",
		"Photometric Target Sphere",
		"Photometric Free Line",
		"Photometric Target Line")
)

--Returns the source class of the super class id 
fn SourceSuperClassID =
(
	#(48, 48, 48, 48, 48, 48, 48, 48,48, 48, 48, 48) --Light types. Note that Target Cylinder and Sphere not exposed
)

--Returns the class id of the source class we use as a source for conversion
fn SourceClassID =
(
	#(#(0x32375fccL, 0xb025cf0L)/* #Free_Light */, 
		#(0x658d4f97L, 0x72cd4259L) /* #Target_Light */,
		#(0x5bcc6d42L, 0xc4f430eL)  /* #Free_Disc */,
		#(0x38732348L, 0x201758b3L) /* #Target_Disc */,
		#(0x36507d92L, 0x105a1a47L) /* #Free_Area */,
		#(0x71794f9dL, 0x70ae52f2L) /* #Target_Area */,
		#(0x46f634e3L, 0x0a327aafL) /* #Free_Cyclinder */,
		#(0x7C8B5B10L, 0x4BD33E86L) /* #Target_Cyclinder */,
		#(0x7ca93582L, 0x1abb6b32L) /* #Free_Sphere */,
		#(0x33FC7AE9L, 0x54433DC5L) /* #Target_Sphere */,
		#(0x78207401L, 0x357f1d58L) /* #Free_Line */,
		#(0x45076885L, 0x40791449L) /* #Target_Line */
		)
)

--Return the name to be display for the destination in the Scene Converter UI - must be in the order than the destination class ids
fn DisplayDestinationName =
(
	#("Arnold Light")
)

--Returns the destination class of the super class id
fn DestinationSuperClassID =
(
	#(48) --Light
)
--Returns the class id of the class we use as a destination for conversion
-- so we convert from SourceClassID() to DestinationClassID()
fn DestinationClassID =
(
		#(0x6705f00dL, 0xca131d05L)/*Arnold Light*/
)

--Validate that the source provided match with one of the SourceClass
fn VerifySource source =
(
	local ret = false	
	local src = SourceClassID()
	
	if(source == undefined) do return false
	
	if(classof src[1] == Array) then
    (
		for cd in src do
		(
			if((cd as string) == (source.ClassID as string)) do
			(
				ret = true
			)
		)
	)
    else
	(
		if((src as string) == (source.ClassID as string)) do
		(
			ret = true
		)
	)
    ret 
)

--Validate that the destination provided match with one of the DestinationClass
fn VerifyDestination destination =
(
	local ret = false	
	local dest = DestinationClassID()

	if(destination == undefined) do return false
	
	if(classof dest[1] == Array) then
    (
		for cd in dest do
		(
			if((cd as string) == (destination.ClassID as string)) do
			(
				ret = true
			)
		)
	)
    else
	(
		if((dest as string) == (destination.ClassID as string)) do
		(
			ret = true
		)
	)
    ret 
)

--Returns some information about this conversion script
fn AboutText =
(
	"Converts from a Photometric Light to an Arnold Light"
)
--This internal function does the parameter mapping
fn ConvertSupportedParameters legacyLight arnoldLight =
(
	if ((false == VerifySource legacyLight) or 
		(false == VerifyDestination arnoldLight)
		) do
	(
		--Not the suitable nodes
		return undefined
	)		
	
	arnoldLight.on = legacyLight.on
	

	/* Color Settings */
	arnoldLight.filter = legacyLight.rgbFilter

	arnoldLight.color = legacyLight.color
	arnoldLight.useColor = not legacyLight.useKelvin
	
	-- Use Kelvin Temp Color?
	arnoldLight.useKelvin = legacyLight.useKelvin
	arnoldLight.kelvin = legacyLight.kelvin	
	
	-- Use Light Color Preset? List doesn't match Photometric, so use resulting color (above)
	--arnoldLight.usePreset =  not legacyLight.useKelvin
	
	-- Shadows
	arnoldLight.shadow_Color = legacyLight.shadowColor
	arnoldLight.density = legacyLight.shadowMultiplier
	arnoldLight.castShadows = legacyLight.castShadows

	/* Calculate intensity */
	local intensity = legacyLight.Intensity
	
	-- If Photometric only
		if(legacyLight.Distribution == 3) do
			intensity = (legacyLight.intensity / legacyLight.OriginalIntensity) * 1000
	
	-- If 'Use Dimmer'
	if(legacyLight.useMultiplier == true) do
		(intensity = intensity * legacyLight.multiplier * 0.01f ) -- Multiplier value is a percentage 
	
	-- We might need to test if this param is valid.
	local exposure_physical_scale = 1500 -- default value used in arnold
	if(SceneExposureControl.exposurecontrol != undefined) do
	(
		exposure_physical_scale = SceneExposureControl.exposurecontrol.physical_scale
	)
	intensity = intensity / exposure_physical_scale
	intensity = intensity * 3.1415926535 -- Zap's magic adjustment
	
	-- Scale to System Units
	-- Get Scene Scale Factor based on units.SystemType to find Units-per-meter
	-- Returns #Inches #Feet #Miles #Millimeters #Centimeters #Meters #Kilometers 
	-- intensity /= (meters_per_internal_unit * meters_per_internal_unit);
	
	local scaleFactor = 0.0
	case units.systemType of
	(
		#Inches: scaleFactor = 0.0254
		#Feet: scaleFactor = 0.3048
		#Millimeters: scaleFactor = 0.001
		#Centimeters: scaleFactor = 0.01
		#Miles: scaleFactor = 1609.34
		#Meters: scaleFactor = 1.0
		#Kilometers: scaleFactor = 1000.0
		default: scaleFactor = 0.0254
	)

	intensity = intensity / (scaleFactor * scaleFactor) 

	-- Default Exposure is 8, so use that for consistency
	intensity = intensity / 256.0 -- Divide by Exposure level of 8
	arnoldLight.exposure = 8.0 -- The default in MAXtoA

	arnoldLight.intensity = intensity

	arnoldLight.LightShapeVisible = true		
	arnoldLight.alwaysVisibleinViewport = false -- This doesn't work as it should at this point, so leaving off.
	
	-- If checkboxes are off, then zero-out the default values
	arnoldLight.diffuse = 1.0	-- Ensure default is set
	arnoldLight.specular = 1.0
	if(false == legacyLight.AmbientOnly) then
	(
		--	Affect Diffuse
		if(legacyLight.affectDiffuse != true) do
		(
			arnoldLight.diffuse = 0.0
		)
	
		-- Affect Specular
		if(legacyLight.affectSpecular != true) do
		(
			arnoldLight.specular = 0.0
		)
	) else
	(
		arnoldLight.diffuse = 0.0
		arnoldLight.specular = 0.0
	)
		
	arnoldLight
)

fn CreateDestinationBasedOnSource legacyLight =
(	
	if (false == VerifySource legacyLight)  do
	(
		--Not the suitable node
		return undefined
	)
	
	local legacyTypePoint = false
	local legacyTypeSphere = false
	local legacyTypeLine = false
	local legacyTypeCylinder = false
	local legacyTypeArea = false
	local legacyTypeDisc  = false
	
	local aiDebug = false	-- Don't print debug info
	
	if aiDebug do print "Processing: " + legacyLight.name -- + " " + legacyLight.type.toString
	
	-- Determine the base Shape of the Light we have in the legacy "Lightscape" light type
	case legacyLight.type of
	(
		#Free_Point: (
			legacyTypePoint = true
			if aiDebug do print "legacyTypePoint = true")
		#Target_Point: (
			legacyTypePoint = true
			if aiDebug do print "legacyTypePoint = true")
		#Free_Sphere: (
			legacyTypeSphere = true
			if aiDebug do print "legacyTypeSphere = true")
		#Target_Sphere: (
			legacyTypeSphere = true
			if aiDebug do print "legacyTypeSphere = true")
		#Free_Cylinder: (
			legacyTypeCylinder = true
			if aiDebug do print "legacyTypeCylinder = true")
		#Target_Cylinder: (
			legacyTypeCylinder = true
			if aiDebug do print "legacyTypeCylinder = true")
		#Free_Rectangle: (
			legacyTypeArea = true
			if aiDebug do print "legacyTypeArea = true")
		#Target_Rectangle: (
			legacyTypeArea = true
			if aiDebug do print "legacyTypeArea = true")
		#Free_Disc: (
			legacyTypeDisc = true
			if aiDebug do print "legacyTypeDisc = true")
		#Target_Disc: (
			legacyTypeDisc = true
			if aiDebug do print "legacyTypeDisc = true")
		#Free_Line: (
			legacyTypeLine = true
			if aiDebug do print "legacyTypeLine = true") 
		#Target_Line: (
			legacyTypeLine = true
			if aiDebug do print "legacyTypeLine = true") 
	)
	
	local hasTarget = ((#Target_Point == legacyLight.type) or 
					(#Target_Sphere == legacyLight.type) or
					(#Target_Rectangle == legacyLight.type) or
					(#Target_Disc == legacyLight.type) or
					(#Target_Disc == legacyLight.type))
	
	if aiDebug do (if (hasTarget == true) then (print "Has Target") else (print "No Target"))
					
	-- Enumerate types
	local ISOTROPIC_DIST = 0 -- Uniform Spherical
	local SPOTLIGHT_DIST = 1
	local DIFFUSE_DIST = 2	-- Uniform Diffuse
	local WEB_DIST = 3	
	
	-- Type Defs for Lightscape Equivalents
	local AI_POINT = 0
	local AI_SPOT = 2
	local AI_QUAD = 3
	local AI_DISC = 4
	local AI_CYLINDER = 5
	local AI_PHOTOMETRIC = 7
	
	-- Local variables
	local POINT_RADIUS = 0.01f
		
	local distType = legacyLight.Distribution
	local aiRadius = POINT_RADIUS
	local aiLength = 0.01f
	local aiWidth = 0.01f
	
	local aiTargetType = 99	-- Default Case
	
	/* Determine what the aiTargetType is for the Arnold Light based on Distribution type */	
	case legacyLight.Distribution of
	(
		WEB_DIST: (
			aiTargetType = AI_PHOTOMETRIC			
			if aiDebug do print "Target is Photometric"
			)
		ISOTROPIC_DIST: (
			if aiDebug do print "Uniform Spherical Distribution type"
			if ((legacyTypePoint == true) or (legacyTypeDisc == true) or (legacyTypeSphere == true)) then
				(aiTargetType = AI_POINT
				if aiDebug do print "Convert to Point") 
			else if ((legacyTypeLine == true) or (legacyTypeArea == true) or (legacyTypeCylinder == true)) then	
				(aiTargetType = AI_CYLINDER
				if aiDebug do print "Convert to Cylinder")
			)
		SPOTLIGHT_DIST: (
			if aiDebug do print "Spotlight Distribution Type"
			if((legacyTypePoint == true) or (legacyTypeDisc == true)) then
			(
				aiTargetType = AI_SPOT
				if aiDebug do print "Convert to Spot"
			)
			else if ((legacyTypeLine == true) or (legacyTypeArea == true)) then
			(
				aiTargetType = AI_QUAD
				if aiDebug do print "Convert to Quad"
			)
			else if (legacyTypeSphere == true) then
			(
				aiTargetType = AI_POINT
				if aiDebug do print "Convert to Point"
			)
			else if (legacyTypeCylinder == true) then
			(
				aiTargetType = AI_CYLINDER
				if aiDebug do print "Convert to Cylinder"
			) 
		)
		DIFFUSE_DIST: (
			if aiDebug do print "Uniform Diffuse Hemispherical Distribution type"
			if((legacyTypePoint == true) or (legacyTypeDisc == true)) then
				(aiTargetType = AI_DISC
					if aiDebug do print "Convert to Disc")
			else if ((legacyTypeLine == true) or (legacyTypeArea == true)) then
				(aiTargetType = AI_QUAD
					if aiDebug do print "Convert to Quad")
			else if (legacyTypeSphere == true) then
				(aiTargetType = AI_POINT
					if aiDebug do print "Convert to Point")
			else if (legacyTypeCylinder == true) then
				(aiTargetType = AI_CYLINDER
					if aiDebug do print "Convert to Cylinder")
		)
	)

	/* Based on Target Type and Shape, Create new ai Light and transfer values */

	local arnoldLight

	case aiTargetType of
	(
		AI_PHOTOMETRIC: (
			arnoldLight = Arnold_Light shapeType: AI_Photometric	
			if(legacyLight.webfile != "" and legacyLight.webfile != undefined) do
				(arnoldLight.file = legacyLight.webfile)			
			arnoldLight.photometricRadius = POINT_RADIUS

			if aiDebug do print "Created Photometric")
		
		AI_SPOT: (
			arnoldLight = Arnold_Light shapeType: AI_Spot
			if(legacyTypePoint == true) then
				(arnoldLight.lightRadius = POINT_RADIUS)
			else if (legacyTypeDisc == true) then
				(arnoldLight.lightRadius = legacyLight.light_radius)			
			arnoldLight.cone_angle = legacyLight.falloff
			arnoldLight.penumbra_angle = legacyLight.falloff - legacyLight.hotSpot	
			
			if aiDebug do print "Created Spot")
		
		AI_POINT: (
			arnoldLight = Arnold_Light shapeType: AI_Point  lightRadius:1			
			if(legacyTypePoint == true) then
				(arnoldLight.lightRadius = POINT_RADIUS)
			else -- if (legacyTypeDisc == true or iSphere == true) then
				(arnoldLight.lightRadius = legacyLight.light_radius) 
			
			if aiDebug do print "Created Point")
		
		AI_QUAD: (
			aiWidth = 0.1f			
			arnoldLight = Arnold_Light shapeType: AI_Quad 			
			/* Defaults */
			arnoldLight.spread = 1.0
			arnoldLight.softEdge = 0.0
			arnoldLight.quadRoundness = 0.0
			
			if(legacyTypeLine == true) then (
				aiLength = legacyLight.light_length
				aiWidth = POINT_RADIUS * 10.0	)
			else ( -- if (legacyTypeArea == true) then 				(
					aiLength = legacyLight.light_length
					aiWidth = legacyLight.light_width )				
			if (distType == SPOTLIGHT_DIST) do (				
					arnoldLight.spread = legacyLight.beam_angle / 180.0f )		
					
			arnoldLight.quadY = aiLength
			arnoldLight.quadX = aiWidth 
					
			arnoldLight.portal = false
					
			if aiDebug do print "Created Quad")
		
		AI_DISC: (
			arnoldLight = Arnold_Light shapeType: AI_Disc	
			if(legacyTypePoint == true) then
				(arnoldLight.lightRadius = POINT_RADIUS)
			else -- if (legacyTypeDisc == true) then
				(arnoldLight.lightRadius = legacyLight.light_radius)			
			arnoldLight.spread = 1.0 
			
			if aiDebug do print "Created Disc")
		
		AI_CYLINDER: (
			arnoldLight = Arnold_Light shapeType: AI_Cylinder
			if(legacyTypeLine == true) then (
					aiLength = legacyLight.light_length
					aiRadius = POINT_RADIUS * 10.0 )
			else if (legacyTypeArea == true) then (
					aiLength = legacyLight.light_length
					aiRadius = legacyLight.light_width  * 0.5 )
			else if (legacyTypeCylinder == true) then (
					aiLength = legacyLight.light_length
					aiRadius = legacyLight.light_radius )					
			arnoldLight.lightRadius = aiRadius
			arnoldLight.height = aiLength 
					
			if aiDebug do print "Created Cylinder")

		default: (
			if aiDebug do (print "Case failed on aiTargetType " + aiTargetType as String)
			return undefined)
	)		

	--if(hasTarget == false) do
		arnoldLight.targDist = 10 -- legacyLight.Target_Distance

	arnoldLight.targeted = hasTarget

	if aiDebug do print "___"
	
	arnoldLight
)

--This function is use as entry when the source is missing (due to a missing plugin) and cannot be completly loaded. 
--In that case the default parameters are loaded.
fn DefaultConversion legacyLight=
(
	--Clone the node to get all the node's data : wirecolor, transform and controllers etc.  
	--to be duplicated to the new node
	
	local ar = #() --the maxOps.CloneNodes needs an array of nodes
	ar[1] = legacyLight
	
	--When you are cloning a light which has a target, the target will be cloned as well
	--That is what the actualNodesList contains, all cloned nodes from the node we provided.
	local actualNodesList = #()
	
	--The new cloned nodes
	local newNodesList 	 = #()
	
	--Clone our node
	local bRes = maxOps.CloneNodes ar actualNodeList:&actualNodesList newNodes:&newNodesList
	if (false == bRes)do return undefined --something went wrong
	
	--Create an Arnold light	
	local arnoldLight = Arnold_Light shapeType: 0  lightRadius:1
	
	--Call the parameters mapping function to convert
	ConvertSupportedParameters legacyLight arnoldLight

	newLight			    = newNodesList[1]
	newLight.name        	= legacyLight.name --copy its name
	newLight.baseobject 	= arnoldLight.baseobject
	
	--Delete the dummy physical light from the scene
	delete arnoldLight
	
	--return the new created light node
	newLight
)

--Main entry point from this script
--This function handles the node's creation
fn Conversion legacyLight =
(
	if (false == VerifySource legacyLight)  do
	(
		--Not the suitable node
		return undefined
	)
	
	--Clone the node to get all the node's data : wirecolor, transform and controllers etc.  
	--to be duplicated to the new node
	
	local ar = #() --the maxOps.CloneNodes needs an array of nodes
	ar[1] = legacyLight
	
	--When you are cloning a light which has a target, the target will be cloned as well
	--That is what the actualNodesList contains, all cloned nodes from the node we provided.
	local actualNodesList = #()
	
	--The new cloned nodes
	local newNodesList 	 = #()
	
	--Clone our node
	local bRes = maxOps.CloneNodes ar actualNodeList:&actualNodesList newNodes:&newNodesList
	if (false == bRes)do return undefined --something went wrong
	
	--Create a Photometric light	
	local arnoldLight = CreateDestinationBasedOnSource legacyLight

	if(arnoldLight == undefined) do return undefined	-- Error in creating new target, so just exit.

	--Call the parameters mapping function to convert
	ConvertSupportedParameters legacyLight arnoldLight

	newLight			    = newNodesList[1]
	newLight.name        	= legacyLight.name --copy its name
	newLight.baseobject 	= arnoldLight.baseobject
		
	--Delete the dummy physical light from the scene
	delete arnoldLight 
	--return the new created light node
	return newLight	
)